#
# GoogleCalendar Copyright (C) Michael Ebert, 2007
#
# This program is free software; you can redistribute it and/or 
# modify it under the terms of the GNU General Public License, version 2.
#

package Plugins::GoogleCalendar::GoogleCalendarAccess;

use strict;
# Get access to the msg and msgf routines
use Slim::Utils::Misc;
use Net::Google::Calendar;
use DateTime::Format::Strptime;

sub new {
	my $class = shift;  
	my ($username, $password, $wholetimeStr) = @_;
	
	my $cal = Net::Google::Calendar->new;
	
	my $self  = {
		'username'	 	=> $username,
		'password'		=> $password,
		'cal'		 			=> $cal,
		'wholetimeStr'=> $wholetimeStr,		
		'isLoggedIn'	=> 0
	};

	bless $self, $class;
	
	$self->login();
	return $self;
}

sub isLoggedIn {
	my $objref=shift;
	return $objref->{"isLoggedIn"};
}

sub login {
	my $objref=shift;
	
	$::d_plugins && msg("Google Calendar: login to google with $objref->{\"username\"} ...\n");
		
	my $login_retval = $objref->{"cal"}->login($objref->{"username"}, $objref->{"password"});
	if ($login_retval == 1) {
		$objref->{"isLoggedIn"} = 1;		
	} else {
		# do login error handling later
		warn() if $@;
		$objref->{"isLoggedIn"} = 0;		
		$::d_plugins && msg("Google Calendar: login wasn't successful ...\n");
	}		
}

sub getTodayAndTomorrowStr {
  my $self = shift;
  my ($offset) = @_;

  my ($sec,$min,$hour,$mday,$mon,$year,$wday,$yday,$isdst)=localtime(time);
  my $start    = sprintf("%4d-%02d-%02d", $year+1900,$mon+1,$mday + $offset);
  # tomorrow exclusive
  my $end = sprintf("%4d-%02d-%02d", $year+1900,$mon+1,$mday + 1 + $offset); 
  
  return $start, $end;
}


sub getCalendarEvents {
  my $self = shift;
  my ($start, $until) = @_;

  my %opts = ();
  if ($start && $until) {
    $opts{'start-min'} = $start;
    $opts{'start-max'} = $until;
  } elsif ($start) {
  	$opts{'start-min'} = $start;
  }
  
  my @events = ();
  for ($self->{"cal"}->get_events(%opts)) {
    my $start;
    my $finish;

    # ugly hack to do exception handling
    # when() function complains if entry doesn't have a start date
    # $@ = error message from the eval command
    eval { ($start, $finish) = $_->when() }; warn $@ if $@;

    my %new_event = (
		     title    => $_->title,
		     desc     => $_->content->body,
		     where    => $_->location->getAttribute("valueString"),
		     datetime => [$start,$finish]
		    );  
    push(@events,\%new_event);
    
    #print "*******************\n\n";
    #print $_->title."\n";
    #print $_->content->body."\n";
    
    #print $_->location->getAttribute("valueString")."\n";
    #print $_->status->getAttribute("value")."\n";
    #print $_->transparency->getAttribute("value")."\n";
    #print $_->visibility->getAttribute("value")."\n";
    
    
    #print "${start}-${finish}\n";
    #print "*******************\n\n";
    
    # print $_->as_xml;
  }
  
  return @events;
}


sub getEventString {
	my $self = shift;
	my $event_ref = @_[0];
	my %event = %$event_ref;	
	my $start  = %event->{'datetime'}[0];	
	my $finish = %event->{'datetime'}[1];	
	
	my $formatter = new DateTime::Format::Strptime(pattern => '%H:%M');
	$start->set_formatter($formatter);
	$finish->set_formatter($formatter);
	
	my $res;
	if ($start->hms eq "00:00:00" && $finish->hms eq "00:00:00") {
			$res = "[" . $self->{'wholetimeStr'} . "] " . %event->{'title'};
	} else {
		 	$res = "[" . $start . "-" . $finish . "] " . %event->{'title'};
	}
	
	return $res;
}


sub getEventStringComplete {
	my $self = shift;
	my $event_ref = @_[0];
	my %event = %$event_ref;	
	my $start  = %event->{'datetime'}[0];	
	my $finish = %event->{'datetime'}[1];	
	my $res;
	
	if ($start && $finish) {
		my $formatter1 = new DateTime::Format::Strptime(pattern => '%F, %H:%M');
		my $formatter2 = new DateTime::Format::Strptime(pattern => '%H:%M');
		my $formatter3 = new DateTime::Format::Strptime(pattern => '%F');
		my $dateonly = $start->clone();
	
		$start->set_formatter($formatter1);		
		$finish->set_formatter($formatter2);
		$dateonly->set_formatter($formatter3);
		
		if ($start->hms eq "00:00:00" && $finish->hms eq "00:00:00") {
				#$res = "[" . $dateonly . ", " . $self->{'wholetimeStr'} . "] " . %event->{'title'};
				$res = "[" . $dateonly . "] " . %event->{'title'};
		} else {
			 	$res = "[" . $start . "-" . $finish . "] " . %event->{'title'};
		}
	} else {
		$res = "[no date] " . %event->{'title'};
	}
	
	return $res;

}


sub getCalendarDataToday {
  my $self=shift;
  my ($today, $tomorrow) = $self->getTodayAndTomorrowStr(0);
  
  # get events for today
  my @events = $self->getCalendarEvents($today, $tomorrow);

  my @displayed_events = ();
  # iterate through all events
  foreach (@events) {    	
  	# get display string for this event
    my $entry = $self->getEventString($_);
    push(@displayed_events, $entry);
  }

	@displayed_events = sort(@displayed_events);
  return @displayed_events;
}


sub getCalendarDataTomorrow {
  my $self=shift;
  my ($start_date, $end_date) = $self->getTodayAndTomorrowStr(1);
  
  my @events = $self->getCalendarEvents($start_date, $end_date);

  my @displayed_events = ();
  # iterate through all events
  foreach (@events) {    	
  	# get display string for this event
    my $entry = $self->getEventString($_);
    push(@displayed_events, $entry);
  }

	@displayed_events = sort(@displayed_events);
  return @displayed_events;
}

sub getCalendarDataAll {
  my $self=shift;
 
 	my ($today, $tomorrow) = $self->getTodayAndTomorrowStr(0);
 	
  my @events = $self->getCalendarEvents($today);

	my @displayed_events = ();
  # iterate through all events
  foreach (@events) {    	
  	# get display string for this event
    my $entry = $self->getEventStringComplete($_);
    push(@displayed_events, $entry);
  }
	
	@displayed_events = sort(@displayed_events);
  return @displayed_events;
}


sub getCalendarDataTodayOneLine {
	my $self=shift;
  
  # get events for today
  my @displayed_events = $self->getCalendarDataToday();

	my $result = "";
	foreach (@displayed_events) { $result .= "            " . $_ };
	
 	return $result;
}

1;

