#
# GoogleCalendar Copyright (C) Michael Ebert, 2007
#
# A Slimserver plugin for displaying Google Calendar Entries
# (see http://www.google.com/calendar/)

# The Plugin uses "Net::Google::Calendar", v0.8
# Copyright Simon Wistow, 2006

# This code is derived from code with the following copyright message:
#
# SlimServer Copyright (C) 2001-2007 Logitech.
# This program is free software; you can redistribute it and/or
# modify it under the terms of the GNU General Public License,
# version 2.
#
#
#	Version History
#
# 0.2   08/29/2007  - Screensaver mode added, shows calendar events of the
#                     current day as ticker line
#	0.1		08/20/2007  - Initial Release
#
#
# FEEDBACK
# Please direct all feedback to katmic on the Slim Devices public forums 
# at forums.slimdevices.com
#

package Plugins::GoogleCalendar::Plugin;


use strict;
# Get access to the msg and msgf routines
use Slim::Utils::Misc;
use Slim::Utils::Strings qw (string);
use Plugins::GoogleCalendar::GoogleCalendarAccess;

our $calaccess = {};
# List of items to display
my @lines = ();
# Hash to store the line that is currently selected on each client
my %linesSelected = ();

my $username = "";
my $password = "";

my $tickerLine = "";

# per-client screensaver state information
my $savers = {};

sub strings {
	return q^
PLUGIN_GOOGLECAL_DISPLAY_NAME
	EN	Google Calendar
	DE	Google Kalender
PLUGIN_GOOGLECAL_TODAY
	EN	Today
	DE	Heute
PLUGIN_GOOGLECAL_TOMORROW
	EN	Tomorrow
	DE	Morgen
PLUGIN_GOOGLECAL_ALL_ENTRIES
	EN	All entries
	DE	Alle Eintrge
PLUGIN_GOOGLECAL_WAIT
	DE	Kalenderdaten werden abgerufen. Bitte warten ...
	EN	Bitte warten, requesting calendar data ...
SETUP_GROUP_PLUGIN_GOOGLECAL
	EN	Access to Google Calendar
	DE	Zugriff auf Google Calendar
SETUP_GROUP_PLUGIN_GOOGLECAL_DESC
	EN	Displays Google calendar entries
	DE	Zeigt Google Kalendereintrge an
SETUP_PLUGIN_GOOGLECAL_USERNAME
	EN	Google Calendar Username
	DE	Google Kalender Benutzername
SETUP_PLUGIN_GOOGLECAL_USERNAME_DESC
	DE	Dein Google Kalender Benutzername
	EN	Your Google Calendar username
SETUP_PLUGIN_GOOGLECAL_PASSWORD
	DE	Google Kalender Passwort
	EN	Google Calendar Password
SETUP_PLUGIN_GOOGLECAL_PASSWORD_DESC
	DE	Dein Google Kalender Passwort
	EN	Your Google Calendar password
SETUP_PLUGIN_GOOGLECAL_PASSWORD_CHANGED
	DE	Passwort von Google-Kalender wurde gendert.
	EN	Google Calendar password changed
MISC_GOOGLECAL_WHOLETIME
	DE	ganztgig
	EN	whole-time
PLUGIN_SCREENSAVER_GOOGLECAL
	DE	Google-Kalender Ticker
	EN	Google Calendar Ticker
SETUP_PLUGIN-GOOGLE-CHECK-WHEN_DESC
	DE	Wie oft wird Google kontaktiert? (in Minuten, Vorgabe 5)
	EN	How often to check Google, in minutes.  Default is 5.
^};


# Called by the server on plugin startup
sub initPlugin {
	# The list of items that will be displayed.  
	@lines = ('PLUGIN_GOOGLECAL_TODAY','PLUGIN_GOOGLECAL_TOMORROW','PLUGIN_GOOGLECAL_ALL_ENTRIES');	
}

sub addMenu {
	return "PLUGINS";
}

# below the functions that the Slimserver calls in my module
sub setMode
{
    my $client = shift;
    my $method = shift;

		$::d_plugins && msg("Google Calendar: in setMode ...\n");
		
		
    # Handle requests to exit this mode/plugin by going back to where the user was before they came
    # here.  If you don't this, pressing LEFT will just put you straight back to where you already
    # are! (try commenting out the following if statement) 
    if ($method eq 'pop') {
			# Pop the current mode off the mode stack and restore the previous one
			Slim::Buttons::Common::popMode($client);
			return;
    }
    
    unless( defined $calaccess->{$client} ) {
	    $calaccess->{$client} = new Plugins::GoogleCalendar::GoogleCalendarAccess($username, $password, $client->string('MISC_GOOGLECAL_WHOLETIME'));	    
    }

    # use INPUT.List to display the list of commands.  This takes several parameters, which are
    # passed to it via a hash reference.  All of INPUT.Lists parameters can be values or
    # references to functions that it should call to get the values.  Each function is called with
    # two parameters: a client reference and the currently selected item in the list of items that
    # it is displaying.
    my %params = (
    	listRef         => \@lines,
			stringExternRef => 1,
			header          => 'PLUGIN_GOOGLECAL_DISPLAY_NAME',
			stringHeader    => 1,
			headerAddCount  => 1,
			callback        => \&googleCalExitHandler,
			overlayRef      => sub { return (undef, Slim::Display::Display::symbol('rightarrow')) },
			overlayRefArgs  => 'C',
    );

    # Use the new mode defined by INPUT.List and let it do all the hard work of displaying the
    # list, moving it up and down, etc, etc.  Do a pushMode, rather than a pushModeLeft as
    # the server will already have called pushModeLeft when it pushed into this plugin's mode.
    # Calling pushModeLeft again would cause the new mode to be displayed the moment the user
    # goes into the plugin by pushing left on the plugins list, then scrolled off the screen to be
    # replaced with itself.     
    Slim::Buttons::Common::pushMode($client, 'INPUT.List', \%params);    
}

    # This is called by the server to work out what to do when the user 
    # presses buttons on the remote.
sub getFunctions {
	# All our button presses are handled by INPUT.List so just return 
        # a reference to an empty hash.
	return {};
}

sub getDisplayName {    
    return 'PLUGIN_GOOGLECAL_DISPLAY_NAME';
}

sub googleCalExitHandler {	
    my ($client,$exittype) = @_;
    
    $exittype = uc($exittype);
    my $selection = ${$client->param('valueRef')}; # menu selection: TODAY, TOMORROW, ALL
    
    if ($exittype eq 'LEFT') {
	    Slim::Buttons::Common::popModeRight($client);
    } elsif ($exittype eq 'RIGHT') {
	    my %params = ();
	    if ($selection eq 'PLUGIN_GOOGLECAL_TODAY') {
	      my @event_list = $calaccess->{$client}->getCalendarDataToday();
		    %params = (
			'header' => 'PLUGIN_GOOGLECAL_DISPLAY_NAME',
			'headerAddCount' => 1,
			'stringHeader' => 1,
			'listRef' => \@event_list,		
			'cursorPos' => 1,					
			);
	    } elsif ($selection eq 'PLUGIN_GOOGLECAL_TOMORROW') {
	       my @event_list = $calaccess->{$client}->getCalendarDataTomorrow();
		    %params = (
			'header' => 'PLUGIN_GOOGLECAL_DISPLAY_NAME',
			'headerAddCount' => 1,
			'stringHeader' => 1,
			'listRef' => \@event_list,		
			'cursorPos' => 1,					
			);
	    } elsif ($selection eq 'PLUGIN_GOOGLECAL_ALL_ENTRIES') {
	       my @event_list = $calaccess->{$client}->getCalendarDataAll();
		    %params = (
			'header' => 'PLUGIN_GOOGLECAL_DISPLAY_NAME',
			'headerAddCount' => 1,
			'stringHeader' => 1,
			'listRef' => \@event_list,	
			'cursorPos' => 1,					
			);
		}			
	    Slim::Buttons::Common::pushModeLeft($client, 'INPUT.List',\%params);	
    } else {
	    return;
    }
}
	
#############################################################
##    Plugin Web GUI
############################################################

sub setupGroup {
	my %setupGroup = (
			  PrefOrder => ['plugin_googlecal_username','plugin_googlecal_password','plugin-google-check-when'],
			  GroupHead => string('SETUP_GROUP_PLUGIN_GOOGLECAL'),
			  GroupDesc => string('SETUP_GROUP_PLUGIN_GOOGLECAL_DESC'),
			  GroupLine => 1,
			  GroupSub => 1,
			  Suppress_PrefSub => 1,
			  Suppress_PrefLine => 1
			 );

        
	my %setupPrefs = (
	'plugin_googlecal_username' => {
		'validate' => \&Slim::Utils::Validate::acceptAll,
		'PrefSize' => 'medium',
	},
	'plugin_googlecal_password' => {
		'validate' => \&Slim::Utils::Validate::acceptAll,
		'changeMsg' => string('SETUP_PLUGIN_GOOGLECAL_PASSWORD_CHANGED'),
		'inputTemplate' => 'setup_input_passwd.html',
		'PrefSize' => 'medium',
	},
	'plugin-google-check-when' => {
		'validate' => \&Slim::Utils::Validate::isInt,
		'PrefSize' => 'medium',
	}
	);
       
	checkDefaults();

	return (\%setupGroup,\%setupPrefs);
}

sub checkDefaults {
	
	if (!Slim::Utils::Prefs::isDefined('plugin_googlecal_username')) {
		Slim::Utils::Prefs::set('plugin_googlecal_username',"")
	}
	if (!Slim::Utils::Prefs::isDefined('plugin_googlecal_password')) {
		Slim::Utils::Prefs::set('plugin_googlecal_password',"")
	}
	if (!Slim::Utils::Prefs::isDefined('plugin-google-check-when')) {
		Slim::Utils::Prefs::set('plugin-google-check-when',5)
	}

	$username = Slim::Utils::Prefs::get('plugin_googlecal_username');
	$password = Slim::Utils::Prefs::get('plugin_googlecal_password');

}

#############################################################
##    Plugin Screensaver Mode
############################################################

sub screenSaver {
	Slim::Utils::Strings::addStrings(strings());
	
	$::d_plugins && msg("Google Calendar: adding screensaver mode\n");
	Slim::Buttons::Common::addSaver(
			'SCREENSAVER.googlecal',
				getScreensaverGoogleCal(),
				\&setScreensaverGoogleCalMode,
				\&leaveScreenSaverGoogleCal,
				'PLUGIN_SCREENSAVER_GOOGLECAL'
	);
}

	our %screensaverGoogleCalFunctions = (
		'done' => sub  {
			my ($client ,$funct ,$functarg) = @_;
	
			Slim::Buttons::Common::popMode($client);
			$client->update();

			# pass along ir code to new mode if requested
			if (defined $functarg && $functarg eq 'passback') {
				Slim::Hardware::IR::resendButton($client);
			}
		},
	);

sub getScreensaverGoogleCal {
	$::d_plugins && msg("Google Calendar: get screensaver \n");
	return \%screensaverGoogleCalFunctions;
}

sub setScreensaverGoogleCalMode() {
	my $client = shift;
	$::d_plugins && msg("Google Calendar: set screensaver mode\n");

	unless( defined $calaccess->{$client} ) {
	    $calaccess->{$client} = new Plugins::GoogleCalendar::GoogleCalendarAccess($username, $password, $client->string('MISC_GOOGLECAL_WHOLETIME'));	    
  }	
  
  # init params
	$savers->{$client} = {
		newevents   => 1,
		line1    		=> 0,
	};

	$client->update( {
			'line' => [ 
				$client->string('PLUGIN_GOOGLECAL_DISPLAY_NAME'),
				$client->string('PLUGIN_GOOGLECAL_WAIT')
			],
		} );
		
	updateTickerLine($client);
	
	Slim::Utils::Timers::setTimer(
		$client, 
		Time::HiRes::time() + 0.5,
		\&tickerUpdateContinue
	);
	
}


sub tickerUpdateContinue {
	my $client = shift;
	
	$::d_plugins && msg("Google Calendar: in tickerUpdateContinue ...\n");
	
	if ( !exists $savers->{$client} ) {
		return;
	}
	
	# add item to ticker	
	$client->update( screensaverGoogleCallines($client) );

	my ($complete, $queue) = $client->scrollTickerTimeLeft();
	my $newevents = $savers->{$client}->{newevents};

	# schedule for next item as soon as queue drains if same feed or after ticker completes if new feed
	my $next = $newevents ? $complete : $queue;

	Slim::Utils::Timers::setTimer(
		$client, 
		Time::HiRes::time() + ( ($next > 1) ? $next : 1),
		\&tickerUpdateContinue
	);
}


# kill tickerUpdate
sub leaveScreenSaverGoogleCal {
	my $client = shift;

	Slim::Utils::Timers::killTimers($client, \&updateTickerLine);
	slim::Utils::Timers::killTimers($client, \&tickerUpdateContinue);
	
	delete $savers->{$client};
	
	$::d_plugins && msg("Google Calendar: Left screensaver mode\n");
}


sub screensaverGoogleCallines {
	my $client = shift;
	
	my ($sec,$min,$hour,$mday,$mon,$year,$wday,$yday,$isdst) = localtime(time);
	my $alarmOn = $client->prefGet("alarm", 0) || $client->prefGet("alarm", $wday);

	my $nextUpdate = $client->periodicUpdateTime();
	Slim::Buttons::Common::syncPeriodicUpdates($client, int($nextUpdate)) if (($nextUpdate - int($nextUpdate)) > 0.01);
	
	return {
		'line'   => ['My Google Calendar'],
		'ticker' => [undef, $tickerLine],
		'overlay'=> [ ($alarmOn ? $client->symbols('bell') : undef) ],
	#	'fonts'  => $fontDef,
	};
}

sub updateTickerLine {
	my $client = shift;
	$::d_plugins && msg("Google Calendar: in updateTickerLine ...\n");
	
	$tickerLine = $calaccess->{$client}->getCalendarDataTodayOneLine();	
	
	# set timer for google event retrieving
	my $mins = Slim::Utils::Prefs::get('plugin-google-check-when');
	if (!defined($mins)) {$mins = 5;};
	
	Slim::Utils::Timers::setTimer($client, Time::HiRes::time() + $mins * 60, \&updateTickerLine);
}


1;

__END__

